<?php
// redeem.php - modified: each code gives 10 loyalty stamps; handles loyalty_prize generation, XP & tiers, and referral flags
require_once __DIR__ . '/scripts/db.php';
session_start();
header('Content-Type: application/json; charset=utf-8');

if (empty($_SESSION['user_id'])) { http_response_code(403); echo json_encode(['ok'=>0,'msg'=>'Login required']); exit; }
$uid = (int)$_SESSION['user_id'];
$code_txt = strtoupper(trim($_POST['code'] ?? ''));
if (!$code_txt || !preg_match('/^[A-Z0-9]{6}$/', $code_txt)) { http_response_code(400); echo json_encode(['ok'=>0,'msg'=>'Invalid code format']); exit; }

try {
    // ensure timezone
    date_default_timezone_set('Asia/Jakarta');
    $pdo->exec("SET time_zone = '+07:00'");

    // find code
    $stmt = $pdo->prepare("SELECT * FROM order_codes WHERE code = ? LIMIT 1");
    $stmt->execute([$code_txt]);
    $code = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$code) { http_response_code(404); echo json_encode(['ok'=>0,'msg'=>'Code not found']); exit; }
    if (!empty($code['used_by'])) { http_response_code(400); echo json_encode(['ok'=>0,'msg'=>'Code already used']); exit; }

    // check valid_date matches today (enforce same-day codes if present)
    $today = date('Y-m-d');
    if (isset($code['valid_date']) && $code['valid_date'] !== $today) {
        http_response_code(400); echo json_encode(['ok'=>0,'msg'=>'Code not valid today']); exit;
    }

    // attempt to add helpful columns if missing (best-effort)
    try {
        // note: some MariaDB versions may not accept IF NOT EXISTS with ADD COLUMN; ignore failure
        $pdo->exec("ALTER TABLE users ADD COLUMN IF NOT EXISTS xp INT DEFAULT 0");
        $pdo->exec("ALTER TABLE users ADD COLUMN IF NOT EXISTS tier VARCHAR(20) DEFAULT 'first'"); // first, classic, ceremonial
        $pdo->exec("ALTER TABLE users ADD COLUMN IF NOT EXISTS referral_flags INT DEFAULT 0");
    } catch (Exception $e) {
        // ignore; columns may already exist or ALTER may not be allowed
    }

    // fetch user row
    $u = $pdo->prepare("SELECT id,email,name,signup_referral_code,referral_flags,xp,tier FROM users WHERE id = ? LIMIT 1");
    $u->execute([$uid]);
    $uRow = $u->fetch(PDO::FETCH_ASSOC);
    if (!$uRow) { http_response_code(500); echo json_encode(['ok'=>0,'msg'=>'User account error']); exit; }

    $pdo->beginTransaction();

    // mark code used (atomic)
    $upd = $pdo->prepare("UPDATE order_codes SET used_by = ?, used_at = NOW() WHERE id = ? AND (used_by IS NULL OR used_by = 0)");
    $upd->execute([$uid, $code['id']]);
    if ($upd->rowCount() < 1) { $pdo->rollBack(); http_response_code(400); echo json_encode(['ok'=>0,'msg'=>'Failed to claim code (maybe taken)']); exit; }

    // insert 10 loyalty stamps (testing mode as requested)
    $insStamp = $pdo->prepare("INSERT INTO stamps (user_id, type, source_code_id, created_at) VALUES (?, 'loyalty', ?, NOW())");
    for ($i=0;$i<3;$i++) { $insStamp->execute([$uid, $code['id']]); }

    // log redemption
    $pdo->prepare("INSERT INTO redeem_logs (user_id, code_id, ip, created_at) VALUES (?, ?, ?, NOW())")->execute([$uid, $code['id'], $_SERVER['REMOTE_ADDR'] ?? '']);

    // Referral handling: if this is user's first purchase and they had signup_referral_code, award referrer referral_flags
    // Detect first purchase: try purchases table, fallback to stamps before this code
    $purchaseCount = 0;
    try {
        $q = $pdo->prepare("SELECT COUNT(*) FROM purchases WHERE user_id = ?");
        $q->execute([$uid]);
        $purchaseCount = (int)$q->fetchColumn();
    } catch (Exception $e) {
        $q = $pdo->prepare("SELECT COUNT(*) FROM stamps WHERE user_id = ? AND source_code_id != ?");
        $q->execute([$uid, $code['id']]);
        $purchaseCount = (int)$q->fetchColumn();
    }

    if ($purchaseCount == 0 && !empty($uRow['signup_referral_code'])) {
        $sref = trim($uRow['signup_referral_code']);
        if ($sref !== '') {
            $qr = $pdo->prepare("SELECT id FROM users WHERE referral_code = ? LIMIT 1");
            $qr->execute([$sref]);
            $found = $qr->fetch(PDO::FETCH_ASSOC);
            if ($found && !empty($found['id'])) {
                $pdo->prepare("UPDATE users SET referral_flags = referral_flags + 1 WHERE id = ?")->execute([$found['id']]);
            }
        }
        // clear signup_referral_code to avoid applying again later
        $pdo->prepare("UPDATE users SET signup_referral_code = '' WHERE id = ?")->execute([$uid]);
    }

    // count total loyalty stamps
    $s2 = $pdo->prepare("SELECT COUNT(*) FROM stamps WHERE user_id = ? AND type = 'loyalty'");
    $s2->execute([$uid]);
    $totalStamps = (int)$s2->fetchColumn();

    $response = ['ok'=>1,'msg'=>'Redeemed successfully. Loyalty stamps added.','added'=>3,'total_stamps'=>$totalStamps];

    // If totalStamps >= 10 -> generate loyalty_prize, reset stamps, award XP & upgrade tier
    if ($totalStamps >= 10) {
        // create unique 6-char code
        $chars = 'ABCDEFGHJKMNPQRSTUVWXYZ23456789';
        do {
            $voucher = '';
            for ($i=0;$i<6;$i++) $voucher .= $chars[random_int(0, strlen($chars)-1)];
            $ck = $pdo->prepare("SELECT id FROM order_codes WHERE code = ? LIMIT 1"); $ck->execute([$voucher]);
            $exists = (bool)$ck->fetch(PDO::FETCH_ASSOC);
        } while ($exists);

        // insert as loyalty_prize (valid today)
        $meta = json_encode(['user'=>$uid,'type'=>'loyalty_prize']);
        $pdo->prepare("INSERT INTO order_codes (code, valid_date, source, category, meta, created_at) VALUES (?, CURDATE(), 'redeem', 'loyalty_prize', ?, NOW())")->execute([$voucher, $meta]);

        // delete loyalty stamps (reset)
        $pdo->prepare("DELETE FROM stamps WHERE user_id = ? AND type = 'loyalty'")->execute([$uid]);

        // Award XP: 10 XP per stamp
        $xpGain = 10 * 10; // award 10 XP per stamp for the 10-stamp reward (10 stamps => 100 XP)
        # ensure xp and tier fields exist - best effort
        try {
            $pdo->exec("ALTER TABLE users ADD COLUMN IF NOT EXISTS xp INT DEFAULT 0");
            $pdo->exec("ALTER TABLE users ADD COLUMN IF NOT EXISTS tier VARCHAR(20) DEFAULT 'first'");
        } catch (Exception $e) { /* ignore */ }

        $u2 = $pdo->prepare("SELECT xp, tier FROM users WHERE id = ? LIMIT 1"); $u2->execute([$uid]); $ur = $u2->fetch(PDO::FETCH_ASSOC);
        $curXp = (int)($ur['xp'] ?? 0); $tier = $ur['tier'] ?? 'first';
        $newXp = $curXp + $xpGain;

        // thresholds: first -> classic at 100, classic -> ceremonial at 500
        if ($tier === 'first' && $newXp >= 100) {
            $tier = 'classic'; $newXp = 100;
        } elseif ($tier === 'classic' && $newXp >= 500) {
            $tier = 'ceremonial'; $newXp = 500;
        }

        $pdo->prepare("UPDATE users SET xp = ?, tier = ? WHERE id = ?")->execute([$newXp, $tier, $uid]);

        $response['special'] = ['voucher_code'=>$voucher, 'type'=>'loyalty_prize'];
        $response['msg'] = 'Congratulations! You reached 10 loyalty stamps and earned a loyalty prize.';
    }

    $pdo->commit();
    echo json_encode($response);
    exit;

} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) $pdo->rollBack();
    http_response_code(500);
    echo json_encode(['ok'=>0,'msg'=>'Server error: '.$e->getMessage()]);
    exit;
}
?>