<?php
// inventory.php - Admin inventory & sales
// Path on server: /home/hiyorima/public_html/hiyorimatcha/priv/admin/inventory.php

// TEMP: show errors while editing (remove or comment out later)
ini_set('display_errors', 1);
error_reporting(E_ALL);

// Load DB (adjust path if your scripts folder differs)
$dbPath = __DIR__ . '/../../scripts/db.php';
if (!file_exists($dbPath)) {
    header('Content-Type: text/plain; charset=utf-8', true, 500);
    echo "Fatal: required file not found: {$dbPath}\nPlease ensure scripts/db.php exists at /public_html/hiyorimatcha/scripts/db.php\n";
    exit;
}
require_once $dbPath;

// basic admin secret check (optional - your db.php may define ADMIN_SECRET constant)
$secret = $_GET['admin_secret'] ?? '';
if (defined('ADMIN_SECRET') && ADMIN_SECRET !== $secret) {
    http_response_code(403);
    echo 'Forbidden';
    exit;
}

// helper: fetch inventory
function get_inventory($pdo) {
    $stmt = $pdo->prepare("SELECT * FROM inventory ORDER BY id DESC");
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Action handlers
$action = $_POST['action'] ?? '';
$msg = '';

// normalize category helper
function normalize_category($val) {
    $val = strtolower(trim($val));
    return $val === 'non-consumable' ? 'non-consumable' : 'consumable';
}

// ADD NEW ITEM
if ($action === 'add_item') {
    $sku = trim($_POST['sku'] ?? '');
    $name = trim($_POST['name'] ?? '');
    $unit = in_array($_POST['unit'] ?? 'g', ['g','ml','pcs']) ? $_POST['unit'] : 'g';
    $qty_on_hand = floatval($_POST['qty_on_hand'] ?? 0);
    $package_qty = floatval($_POST['package_qty'] ?? 0);
    $package_cost = floatval($_POST['package_cost'] ?? 0);
    $category = normalize_category($_POST['category'] ?? 'consumable');

    if ($package_qty > 0 && $package_cost > 0) {
        $unit_cost = $package_cost / $package_qty;
    } else {
        $unit_cost = floatval($_POST['unit_cost'] ?? 0);
    }

    $threshold_pct = ($category === 'consumable') ? 0.20 : 0.12;
    $reorder_threshold = round($qty_on_hand * $threshold_pct, 3);

    if ($sku === '' || $name === '') {
        $msg = 'SKU and Name are required';
    } else {
        $stmt = $pdo->prepare("INSERT INTO inventory (sku,name,unit,qty_on_hand,package_qty,package_cost,unit_cost,category,reorder_threshold) VALUES (?,?,?,?,?,?,?,?,?)");
        $stmt->execute([$sku,$name,$unit,$qty_on_hand,$package_qty,$package_cost,$unit_cost,$category,$reorder_threshold]);
        $msg = 'Item added';
    }
}

// TOP-UP (weighted average unit cost)
if ($action === 'topup') {
    $id = intval($_POST['id'] ?? 0);
    $add = floatval($_POST['add_qty'] ?? 0);
    $package_qty = floatval($_POST['package_qty'] ?? 0);
    $package_cost = floatval($_POST['package_cost'] ?? 0);
    $unit_cost_override = floatval($_POST['unit_cost_override'] ?? 0);

    $stmt = $pdo->prepare("SELECT qty_on_hand, unit_cost, category, package_qty, package_cost FROM inventory WHERE id = ? LIMIT 1");
    $stmt->execute([$id]);
    $r = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$r) {
        $msg = 'Item not found';
    } else {
        $currentQty = floatval($r['qty_on_hand']);
        $currentUnitCost = floatval($r['unit_cost']);
        $category = $r['category'] ?? 'consumable';

        if ($package_qty > 0 && $package_cost > 0) {
            $added_unit_cost = $package_cost / $package_qty;
        } elseif ($unit_cost_override > 0) {
            $added_unit_cost = $unit_cost_override;
        } else {
            $added_unit_cost = $currentUnitCost;
        }

        $totalQty = $currentQty + $add;
        if ($totalQty > 0) {
            $new_unit_cost = (($currentQty * $currentUnitCost) + ($add * $added_unit_cost)) / $totalQty;
        } else {
            $new_unit_cost = $added_unit_cost;
        }

        $threshold_pct = ($category === 'consumable') ? 0.20 : 0.12;
        $new_reorder = round($totalQty * $threshold_pct, 3);

        $stmt = $pdo->prepare("UPDATE inventory SET qty_on_hand = ?, unit_cost = ?, package_qty = GREATEST(package_qty,?), package_cost = GREATEST(package_cost,?), reorder_threshold = ? WHERE id = ?");
        $stmt->execute([$totalQty, $new_unit_cost, $package_qty, $package_cost, $new_reorder, $id]);

        $msg = 'Stock updated (new qty: ' . $totalQty . ', unit cost: ' . round($new_unit_cost,2) . ')';
    }
}

// RECORD SALE (menu recipe or raw SKU)
if ($action === 'record_sale') {
    $sale_sku = trim($_POST['sale_sku'] ?? '');
    $qty = floatval($_POST['sale_qty'] ?? 0);
    $sell_price = floatval($_POST['sell_price'] ?? 0);
    $platform = $_POST['platform'] ?? 'grab';
    $platform_fee_percent = floatval($_POST['platform_fee_percent'] ?? 0);
    $packaging_cost = floatval($_POST['packaging_cost'] ?? 0);

    try {
        $pdo->beginTransaction();

        // Try recipe first
        $stmt = $pdo->prepare("SELECT portion FROM menu_items WHERE sku = ? LIMIT 1");
        $stmt->execute([$sale_sku]);
        $menuRow = $stmt->fetch(PDO::FETCH_ASSOC);

        $materials_cost = 0.0;
        if ($menuRow && !empty($menuRow['portion'])) {
            $portion = json_decode($menuRow['portion'], true);
            if (!is_array($portion)) $portion = [];
            foreach ($portion as $ing) {
                $ing_sku = $ing['sku'] ?? null;
                $ing_qty_per_serving = floatval($ing['qty'] ?? 0);
                if (!$ing_sku || $ing_qty_per_serving <= 0) continue;
                $stmt2 = $pdo->prepare("SELECT unit_cost, qty_on_hand FROM inventory WHERE sku = ? LIMIT 1");
                $stmt2->execute([$ing_sku]);
                $inv = $stmt2->fetch(PDO::FETCH_ASSOC);
                $unit_cost = $inv ? floatval($inv['unit_cost']) : 0;
                $materials_cost += $unit_cost * ($ing_qty_per_serving * $qty);

                // decrement inventory
                $stmt3 = $pdo->prepare("UPDATE inventory SET qty_on_hand = qty_on_hand - ? WHERE sku = ?");
                $stmt3->execute([$ing_qty_per_serving * $qty, $ing_sku]);
            }
        } else {
            // raw SKU sale
            $stmt = $pdo->prepare("SELECT qty_on_hand, unit_cost FROM inventory WHERE sku = ? FOR UPDATE");
            $stmt->execute([$sale_sku]);
            $inv = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$inv) throw new Exception('SKU not found for sale');
            if (floatval($inv['qty_on_hand']) < $qty) throw new Exception('Not enough stock for sale');
            $unit_cost = floatval($inv['unit_cost']);
            $materials_cost = $unit_cost * $qty;
            $stmt = $pdo->prepare("UPDATE inventory SET qty_on_hand = qty_on_hand - ? WHERE sku = ?");
            $stmt->execute([$qty, $sale_sku]);
        }

        $platform_fee_amount = ($platform_fee_percent/100.0) * $sell_price;
        $total_cost = $materials_cost + $packaging_cost + $platform_fee_amount;
        $profit = $sell_price - $total_cost;

        $stmt = $pdo->prepare("INSERT INTO sales (user_id, sku, qty, sell_price, platform, platform_fee_percent, platform_fee_amount, packaging_cost, total_cost, profit) VALUES (NULL, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([$sale_sku, $qty, $sell_price, $platform, $platform_fee_percent, $platform_fee_amount, $packaging_cost, $total_cost, $profit]);

        $pdo->commit();
        $msg = 'Sale recorded (profit: ' . round($profit,0) . ')';
    } catch (Exception $e) {
        $pdo->rollBack();
        $msg = 'Error: ' . $e->getMessage();
    }
}

// CSV export
if (isset($_GET['export']) && $_GET['export'] === 'sales') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=sales_export_' . date('Ymd_His') . '.csv');
    $out = fopen('php://output', 'w');
    fputcsv($out, ['id','created_at','user_id','sku','qty','sell_price','platform','platform_fee_amount','packaging_cost','total_cost','profit','notes']);
    $stmt = $pdo->prepare("SELECT id, created_at, user_id, sku, qty, sell_price, platform, platform_fee_amount, packaging_cost, total_cost, profit, notes FROM sales ORDER BY id DESC");
    $stmt->execute();
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) fputcsv($out, $row);
    fclose($out);
    exit;
}

// Fetch inventory for UI
$inventory = get_inventory($pdo);
?>
<!doctype html>
<html><head><meta charset="utf-8"><meta name="viewport" content="width=device-width,initial-scale=1"><title>Inventory — Admin</title>
<style>
body{font-family:Inter,system-ui,Arial,sans-serif;background:#fff;color:#222}
.wrap{max-width:1100px;margin:28px auto;padding:18px}
table{width:100%;border-collapse:collapse;margin-top:12px}
table th, table td{padding:8px;border:1px solid #eee;text-align:left}
.admin-nav a{margin-right:12px;color:#2f6a45;text-decoration:none}
input, select{padding:6px;border:1px solid #ddd;border-radius:6px;width:100%}
button{padding:8px 12px;background:#2f6a45;color:#fff;border:none;border-radius:6px;cursor:pointer}
h1,h2{color:#2f6a45}
.muted{color:#666;font-size:0.95rem}
.notice{background:#fff6f0;padding:10px;border-left:4px solid #ffb3a7}
</style>
</head><body>
<div class="wrap">
  <h1>Inventory & Sales</h1>
  <p class="muted"><?php echo htmlspecialchars($msg) ?></p>

  <section style="display:flex;gap:20px;flex-wrap:wrap">
    <div style="flex:1;min-width:320px">
      <h2>Add new item</h2>
      <form method="post">
        <input type="hidden" name="action" value="add_item" />
        <div><label>SKU</label><input name="sku" required /></div>
        <div><label>Name</label><input name="name" required /></div>
        <div><label>Unit</label>
          <select name="unit">
            <option value="g">g</option>
            <option value="ml">ml</option>
            <option value="pcs">pcs</option>
          </select>
        </div>
        <div><label>Qty on hand</label><input name="qty_on_hand" value="0" /></div>
        <div><label>Package qty (package size, e.g. 50 if 50g)</label><input name="package_qty" value="0" /></div>
        <div><label>Package cost (Rp)</label><input name="package_cost" value="0" /></div>
        <div><label>Unit cost override (Rp per unit) — optional</label><input name="unit_cost" value="0" /></div>
        <div><label>Category</label>
          <select name="category">
            <option value="consumable">Consumable (20% threshold)</option>
            <option value="non-consumable">Non-consumable (12% threshold)</option>
          </select>
        </div>
        <div style="margin-top:8px"><button>Add Item</button></div>
      </form>
    </div>

    <div style="flex:1;min-width:320px">
      <h2>Top-up stock</h2>
      <form method="post">
        <input type="hidden" name="action" value="topup" />
        <div><label>Item</label>
          <select name="id">
            <?php foreach($inventory as $it): ?>
              <option value="<?php echo $it['id'] ?>"><?php echo htmlspecialchars($it['sku'].' — '.$it['name'].' ('.($it['qty_on_hand']).' '.$it['unit'].')')?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div><label>Add qty</label><input name="add_qty" value="0" /></div>
        <div><label>Package qty (if adding a packaged purchase)</label><input name="package_qty" value="0" /></div>
        <div><label>Package cost (Rp)</label><input name="package_cost" value="0" /></div>
        <div><label>Unit cost override (Rp/unit)</label><input name="unit_cost_override" value="0" /></div>
        <div style="margin-top:8px"><button>Update Stock</button></div>
      </form>
    </div>

    <div style="flex:1;min-width:320px">
      <h2>Record sale</h2>
      <form method="post">
        <input type="hidden" name="action" value="record_sale" />
        <div><label>SKU (menu sku like PRODUCT1 or raw SKU)</label>
          <select name="sale_sku">
            <?php foreach($inventory as $it): ?>
              <option value="<?php echo htmlspecialchars($it['sku']) ?>"><?php echo htmlspecialchars($it['sku'].' — '.$it['name'].' ('.($it['qty_on_hand']).' '.$it['unit'].')')?></option>
            <?php endforeach; ?>
            <?php
              // also list menu items if present
              $mstm = $pdo->prepare("SELECT sku,name FROM menu_items ORDER BY id DESC");
              $mstm->execute(); $menus = $mstm->fetchAll(PDO::FETCH_ASSOC);
              foreach($menus as $m){ echo '<option value="'.htmlspecialchars($m['sku']).'">'.htmlspecialchars($m['sku'].' — '.$m['name']).'</option>'; }
            ?>
          </select>
        </div>
        <div><label>Qty (servings)</label><input name="sale_qty" value="1" /></div>
        <div><label>Sell price (Rp)</label><input name="sell_price" value="25000" /></div>
        <div><label>Platform</label>
          <select name="platform">
            <option value="grab">Grab</option>
            <option value="gojek">Gojek</option>
            <option value="offline">Offline</option>
          </select>
        </div>
        <div><label>Platform fee %</label><input name="platform_fee_percent" value="50" /></div>
        <div><label>Packaging cost (Rp)</label><input name="packaging_cost" value="1850" /></div>
        <div style="margin-top:8px"><button>Record Sale</button></div>
      </form>
    </div>

  </section>

  <section style="margin-top:18px">
    <h2>Inventory list</h2>
    <table>
      <thead><tr><th>SKU</th><th>Name</th><th>Qty</th><th>Unit</th><th>Unit cost (Rp)</th><th>Reorder</th></tr></thead>
      <tbody>
        <?php foreach($inventory as $it): ?>
          <tr<?php if ($it['qty_on_hand'] <= $it['reorder_threshold']) echo ' class="notice"';?>>
            <td><?php echo htmlspecialchars($it['sku'])?></td>
            <td><?php echo htmlspecialchars($it['name'])?></td>
            <td><?php echo htmlspecialchars($it['qty_on_hand'])?></td>
            <td><?php echo htmlspecialchars($it['unit'])?></td>
            <td><?php echo htmlspecialchars(number_format($it['unit_cost'],0,',','.'))?></td>
            <td><?php echo htmlspecialchars($it['reorder_threshold'])?></td>
          </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
    <div style="margin-top:8px">
      <a href="?admin_secret=<?php echo urlencode($secret)?>&export=sales">Export sales CSV</a>
    </div>
  </section>
</div>

<script>
  // default platform fee percentages (editable here)
  const platformDefaults = { grab: 50, gojek: 40, offline: 0 };

  const platformSelect = document.querySelector('select[name="platform"]');
  const feeInput = document.querySelector('input[name="platform_fee_percent"]');

  if (platformSelect && feeInput) {
    platformSelect.addEventListener('change', (e) => {
      const v = e.target.value;
      feeInput.value = platformDefaults[v] !== undefined ? platformDefaults[v] : 0;
    });
    feeInput.value = platformDefaults[platformSelect.value] ?? 0;
  }
</script>
</body></html>
