<?php
// account.php - full replacement
require_once __DIR__ . '/scripts/db.php';
if (session_status() !== PHP_SESSION_ACTIVE) session_start();

// simple helper to fetch current user safely
function fetch_current_user($pdo, $uid) {
    $stmt = $pdo->prepare("SELECT id,email,name,referral_code,signup_referral_code,phone,created_at FROM users WHERE id = ? LIMIT 1");
    $stmt->execute([$uid]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

if (empty($_SESSION['user_id'])) {
    header('Location: /hiyorimatcha/auth/login.php');
    exit;
}

$uid = (int)$_SESSION['user_id'];
$user = fetch_current_user($pdo, $uid);
if (!$user) {
    // invalid session user - logout
    header('Location: /hiyorimatcha/auth/logout.php');
    exit;
}

// --- calculate counts and states (server-side) ---
try {
    // stamps counts
    $stmt = $pdo->prepare("SELECT type, COUNT(*) as c FROM stamps WHERE user_id = ? GROUP BY type");
    $stmt->execute([$uid]);
    $stamp_counts = ['loyalty' => 0, 'referral' => 0];
    while ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $stamp_counts[$r['type']] = (int)$r['c'];
    }
    $loyaltyCount = $stamp_counts['loyalty'];
    $referralCount = $stamp_counts['referral'];

    // coins: 1 coin per 2 loyalty stamps, spent coins counted in draws.used_coin
    $coins_generated = intdiv($loyaltyCount, 2);
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM draws WHERE user_id = ? AND used_coin = 1");
    $stmt->execute([$uid]);
    $coins_spent = (int)$stmt->fetchColumn();
    $coins = max(0, $coins_generated - $coins_spent);

    // draws / guarantee computation
    $stmt = $pdo->prepare("SELECT won, attempt_no, created_at, prize FROM draws WHERE user_id = ? ORDER BY id DESC LIMIT 10");
    $stmt->execute([$uid]);
    $recent_draws = $stmt->fetchAll(PDO::FETCH_ASSOC);
    // compute attempts since last win
    $since = 0;
    foreach ($recent_draws as $row) {
        if ((int)$row['won'] === 1) break;
        $since++;
    }
    // For guarantee we consider full history, but simpler: count consecutive losses since last win
    // To give guarantee left we look up until last win across all draws:
    $stmt = $pdo->prepare("SELECT id,won FROM draws WHERE user_id = ? ORDER BY id DESC");
    $stmt->execute([$uid]);
    $all = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $since_all = 0;
    foreach ($all as $r) {
        if ((int)$r['won'] === 1) break;
        $since_all++;
    }
    $next_guarantee_left = max(0, 5 - $since_all);

    // purchases count
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM purchases WHERE user_id = ?");
    $stmt->execute([$uid]);
    $purchases_count = (int)$stmt->fetchColumn();

    // whether user signed up with referral code
    $hasSignupReferral = !empty($user['signup_referral_code']) ? true : false;

    // prepare recent draws JSON for front-end
    $recent_json = json_encode($recent_draws ?: []);

} catch (Exception $e) {
    // if DB error occurs, show readable error (during testing). In production you may hide this.
    http_response_code(500);
    echo "Server error: " . htmlspecialchars($e->getMessage());
    exit;
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1" />
  <title>Account — Hiyori Artisan Matcha</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet" />
  <style>
    html { scroll-behavior: smooth; }
    body { -webkit-font-smoothing:antialiased; -moz-osx-font-smoothing:grayscale; }
    .stat-number { font-size:1.6rem; color:#2f6a45; font-weight:700; }
    /* small modal */
    .modal-bg { background: rgba(0,0,0,0.45); }
  </style>
</head>
<body class="bg-white text-gray-800">

  <!-- ================= HEADER (consistent - use same block across pages) ================= -->
  <header class="fixed top-0 left-0 w-full z-40 bg-white/90 backdrop-blur border-b border-[#E6F0E8]">
    <div class="max-w-6xl mx-auto px-6">
      <nav class="flex items-center justify-between h-20">
        <a href="/hiyorimatcha/index.php" class="text-2xl tracking-widest font-serif text-[#355f44]">HIYORI</a>
        <ul class="hidden md:flex gap-8 items-center text-gray-700">
          <li><a href="/hiyorimatcha/index.php" class="hover:text-black transition">Home</a></li>
          <li><a href="/hiyorimatcha/index.php#story" class="hover:text-black transition">Story</a></li>
          <li><a href="/hiyorimatcha/index.php#menu" class="hover:text-black transition">Menu</a></li>
          <li><a href="/hiyorimatcha/index.php#contact" class="hover:text-black transition">Contact</a></li>
        </ul>
        <div class="flex items-center gap-3">
          <a href="/hiyorimatcha/account.php" class="text-sm px-3 py-2 border rounded-md hidden md:inline">Account</a>
          <a href="/hiyorimatcha/auth/logout.php" class="text-sm px-3 py-2 bg-[#4A7C59] text-white rounded-md">Logout</a>
        </div>
      </nav>
    </div>
  </header>

  <main class="pt-28 max-w-6xl mx-auto px-6 pb-12">
    <section class="bg-[#F7F8F3] rounded-2xl p-8 shadow-md" data-aos="fade-up">
      <div class="flex flex-col md:flex-row items-start justify-between gap-6">
        <div class="flex-1">
          <?php
// compute a nicely formatted display name
$displayName = '';
if (!empty($user['name'])) {
    // ensure lower-case then capitalize first letter of each word, multibyte-safe
    $displayName = mb_convert_case(mb_strtolower($user['name'], 'UTF-8'), MB_CASE_TITLE, 'UTF-8');
} else {
    $displayName = $user['email'];
}
?>
<h1 class="text-4xl md:text-5xl font-serif font-extrabold text-[#365f42]">
  Welcome back, <span id="uname"><?=htmlspecialchars($displayName)?></span>
</h1>

          <p class="mt-2 text-lg text-gray-600">
            Your referral code: <span id="uRef" class="font-mono mx-2 text-[#2f6a45]"><?=htmlspecialchars($user['referral_code'])?></span>
          </p>

          <?php if ($hasSignupReferral && $purchases_count === 0): ?>
            <div class="mt-4 text-sm text-[#2f6a45] font-medium">
              You have 1 bonus loyalty stamp waiting — complete your first order to claim it.
            </div>
          <?php else: ?>
            <div id="signupReferralNote" class="mt-4 text-sm text-gray-700"></div>
          <?php endif; ?>

          <div id="referrerNote" class="mt-3 text-sm text-gray-700">
            You will receive 1 referral stamp when your referred user completes their first order. Each referral stamp can be converted to 1 loyalty stamp after you make a purchase.
          </div>
        </div>

        <div class="w-full md:w-72">
          <div class="bg-white rounded-lg p-4 border">
            <div class="flex items-center justify-between">
              <div>
                <div class="text-sm text-gray-500">Loyalty stamps</div>
                <div id="loyaltyCount" class="stat-number"><?= $loyaltyCount ?></div>
              </div>
              <div>
                <div class="text-sm text-gray-500">Referral stamps</div>
                <div id="referralCount" class="stat-number"><?= $referralCount ?></div>
              </div>
            </div>
            <div class="text-xs text-gray-500 mt-3">
              Terms: Referral stamps are given to referrer when referred user completes first purchase. Referral stamps convert to 1 loyalty stamp when the referrer completes a purchase.
            </div>
          </div>
        </div>
      </div>

      <!-- Redeem + Lucky Draw area -->
      <div class="mt-8 grid md:grid-cols-2 gap-6">
        <!-- redeem -->
        <div>
          <div class="text-lg font-medium text-gray-700">Redeem a code for a loyalty stamp</div>
          <div class="mt-4 bg-white p-4 rounded-lg border">
            <p class="text-sm text-gray-600">Enter your 6-character code.</p>
            <form id="redeemForm" class="mt-3 flex gap-3">
              <input id="redeemInput" type="text" maxlength="6" placeholder="ABC11N" class="px-3 py-2 border rounded-md w-full" />
              <button id="redeemSubmit" type="submit" class="px-4 py-2 bg-[#4A7C59] text-white rounded-md">Redeem</button>
            </form>
            <div id="redeemMsg" class="mt-2 text-sm"></div>
            <details class="mt-4 text-sm text-gray-600 bg-white p-3 rounded-md border">
              <summary class="cursor-pointer">Terms & mechanism</summary>
              <div class="mt-2 text-xs text-gray-600">
                - New user who signed up with referral code receives 1 bonus loyalty stamp after completing their first purchase.<br>
                - Referrer receives 1 referral stamp when their referred user completes first purchase; that referral stamp can be converted to 1 loyalty stamp when the referrer makes a purchase. Only one conversion per purchase.
              </div>
            </details>
          </div>
        </div>

        <!-- lucky draw -->
        <div>
          <div class="text-lg font-medium text-gray-700">Lucky Draw</div>
          <div class="mt-4 bg-white p-4 rounded-lg border">
            <div class="mb-3">Coins: <span id="coins"><?= $coins ?></span></div>
            <div id="coinsNote" class="mb-3 text-sm text-gray-500">Collect 2 loyalty stamps to get 1 coin.</div>
            <div class="flex gap-3 items-center">
              <button id="playBtn" class="px-4 py-3 bg-[#4A7C59] text-white rounded-md disabled:opacity-60">Play Lucky Draw</button>
              <div id="guaranteeInfo" class="text-sm text-gray-600">Guaranteed in <span id="guaranteeLeft"><?= $next_guarantee_left ?></span> tries</div>
            </div>

            <div id="drawHistory" class="mt-4 text-sm text-gray-600">
              <?php
                if (!empty($recent_draws)) {
                    $hist = [];
                    foreach ($recent_draws as $r) {
                        $hist[] = 'Attempt #' . ((int)$r['attempt_no']) . (((int)$r['won']) ? ' - WIN' : ' - lose');
                    }
                    echo implode(' • ', $hist);
                }
              ?>
            </div>

            <div id="prizeCodeArea" class="mt-4 <?php echo ''; ?>" style="<?php echo (false ? '' : 'display:none;'); ?>">
              <strong>Freebies code:</strong> <span id="prizeCode" class="font-mono"></span>
              <div class="text-sm text-gray-500 mt-1">Show this code to admin to claim your prize.</div>
            </div>
          </div>
        </div>

      </div>
    </section>
  
<!-- Tier & XP card (inserted) -->
<div class="mt-6 bg-white rounded-lg p-4 border">
  <div class="flex items-center justify-between">
    <div>
      <div class="text-sm text-gray-500">Tier</div>
      <div id="tierLabel" class="text-xl font-semibold text-[#2f6a45]"><?= htmlspecialchars(ucfirst($user['tier'] ?? 'first')) ?></div>
    </div>
    <div class="text-right">
      <div class="text-sm text-gray-500">XP</div>
      <div id="xpLabel" class="text-lg font-medium"><?= (int)($user['xp'] ?? 0) ?></div>
    </div>
  </div>
  <?php
    $tier = $user['tier'] ?? 'first';
    $xp = (int)($user['xp'] ?? 0);
    $threshold = ($tier === 'first') ? 100 : (($tier === 'classic') ? 500 : 999999);
    $pct = ($threshold > 0) ? min(100, round($xp / $threshold * 100)) : 100;
  ?>
  <div class="mt-4">
    <div class="w-full bg-gray-200 rounded-full h-3 overflow-hidden">
      <div class="h-3 rounded-full" style="width: <?= $pct ?>%; background: <?= ($tier === 'ceremonial') ? '#2b6a3a' : (($tier === 'classic') ? '#4a7c59' : '#a7c79a') ?>;"></div>
    </div>
    <div class="mt-2 text-sm text-gray-600"><?= $xp ?> / <?= $threshold ?> XP</div>
  </div>
  <details class="mt-3 text-sm text-gray-600">
    <summary>Tier benefits</summary>
    <div class="mt-2">
      <strong>First Bloom</strong>: No bonus for now.<br>
      <strong>Hiyori Classic</strong>: 10% chance to get an additional loyalty stamp on redeem.<br>
      <strong>Ceremonial</strong>: 20% chance to get an additional loyalty stamp on redeem.
    </div>
  </details>
</div>

</main>

  <!-- FOOTER (non-sticky) -->
  <footer class="bg-[#F7F8F3] border-t mt-12">
    <div class="max-w-6xl mx-auto px-6 py-8">
      <div class="flex flex-col md:flex-row items-center justify-between gap-6">
        <div class="text-sm text-gray-700">
          © 2025 Hiyori Artisan Matcha — All rights reserved.
        </div>
        <div class="flex items-center gap-6 text-gray-700">
          <div class="flex items-center gap-2">
            <!-- phone icon -->
            <svg width="18" height="18" viewBox="0 0 24 24" fill="none" class="text-gray-600"><path d="M22 16.92v3a2 2 0 0 1-2.18 2 19.86 19.86 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6A19.86 19.86 0 0 1 2.09 4.18 2 2 0 0 1 4 2h3a2 2 0 0 1 2 1.72c.12.9.37 1.77.73 2.59a2 2 0 0 1-.45 2.11L8.91 9.91a16 16 0 0 0 6 6l1.45-1.45a2 2 0 0 1 2.11-.45c.82.36 1.69.61 2.59.73A2 2 0 0 1 22 16.92z" stroke="currentColor" stroke-width="1.2" stroke-linecap="round" stroke-linejoin="round"/></svg>
            <a href="tel:+6287873640479" class="hover:text-black">+6287873640479</a>
          </div>
          <div class="flex items-center gap-2">
            <!-- instagram icon -->
            <svg width="18" height="18" viewBox="0 0 24 24" fill="none"><rect x="2" y="2" width="20" height="20" rx="5" stroke="currentColor" stroke-width="1.2"/><path d="M16 11.37A4 4 0 1 1 12.63 8 4 4 0 0 1 16 11.37z" stroke="currentColor" stroke-width="1.2" stroke-linecap="round" stroke-linejoin="round"/><path d="M17.5 6.5h.01" stroke="currentColor" stroke-width="1.6" stroke-linecap="round" stroke-linejoin="round"/></svg>
            <a href="https://instagram.com/hiyorimatcha" target="_blank" class="hover:text-black">@hiyorimatcha</a>
          </div>
        </div>
      </div>
    </div>
  </footer>

  <!-- Draw modal -->
  <div id="drawModal" class="fixed inset-0 modal-bg hidden items-center justify-center z-50">
    <div class="bg-white rounded-2xl p-6 w-80 text-center shadow-lg">
      <div id="drawModalContent" class="mb-4"></div>
      <button id="drawModalClose" class="px-4 py-2 bg-[#4A7C59] text-white rounded-md">Close</button>
    </div>
  </div>

<script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
<script>
AOS.init({ duration:900, once:true });

// prefilled server values
const pre = {
  coins: <?= json_encode($coins) ?>,
  loyalty: <?= json_encode($loyaltyCount) ?>,
  referral: <?= json_encode($referralCount) ?>,
  guaranteeLeft: <?= json_encode($next_guarantee_left) ?>,
  recent: <?= $recent_json ?>,
  hasSignupReferral: <?= $hasSignupReferral ? 'true' : 'false' ?>,
  purchasesCount: <?= $purchases_count ?>
};

// update UI from server values
function applyPrefill(){
  document.getElementById('coins').textContent = pre.coins;
  document.getElementById('loyaltyCount').textContent = pre.loyalty;
  document.getElementById('referralCount').textContent = pre.referral;
  document.getElementById('guaranteeLeft').textContent = pre.guaranteeLeft;
  if (pre.hasSignupReferral && pre.purchasesCount === 0) {
    // message already printed server-side for new signups; if not, we can set it here too.
  }
  if (pre.recent && pre.recent.length) {
    const hist = pre.recent.map(x => 'Attempt #' + x.attempt_no + (x.won ? ' - WIN' : ' - lose'));
    document.getElementById('drawHistory').textContent = hist.join(' • ');
  }
}
applyPrefill();

// Redeem handler
document.getElementById('redeemForm').addEventListener('submit', async (e)=>{
  e.preventDefault();
  const code = document.getElementById('redeemInput').value.trim().toUpperCase();
  const msgEl = document.getElementById('redeemMsg');
  if (!/^[A-Z0-9]{6}$/.test(code)) { msgEl.innerHTML = '<span class="text-red-600">Invalid code format</span>'; return; }
  document.getElementById('redeemSubmit').disabled = true;
  msgEl.innerHTML = 'Checking...';
  try {
    const resp = await fetch('/hiyorimatcha/redeem.php', {
      method:'POST',
      headers: {'Content-Type':'application/x-www-form-urlencoded'},
      body: 'code=' + encodeURIComponent(code)
    });
    const text = await resp.text();
    let j;
    try { j = JSON.parse(text); } catch(e) { j = { ok:0, msg: text }; }
    if (j.ok) {
      msgEl.innerHTML = '<span class="text-green-700">' + (j.msg || 'Redeemed') + '</span>';
      document.getElementById('redeemInput').value = '';
      // refresh simple UI values by reloading the page or calling ajax_user_data.php
      setTimeout(()=> location.reload(), 800);
    } else {
      msgEl.innerHTML = '<span class="text-red-600">' + (j.msg || 'Error') + '</span>';
    }
  } catch (err) {
    msgEl.innerHTML = '<span class="text-red-600">Network error</span>';
  } finally {
    document.getElementById('redeemSubmit').disabled = false;
  }
});

// Play button: client side cooldown + server call
let lastPlayTs = 0;
document.getElementById('playBtn').addEventListener('click', async ()=>{
  const now = Date.now();
  if (now - lastPlayTs < 3000) {
    const modalContent = document.getElementById('drawModalContent');
    modalContent.innerHTML = '<div class="text-sm text-gray-600">Please wait 3 seconds before trying again.</div>';
    const modal = document.getElementById('drawModal');
    modal.classList.remove('hidden'); modal.style.display='flex';
    return;
  }
  lastPlayTs = now;

  const btn = document.getElementById('playBtn');
  btn.disabled = true;
  const modalContent = document.getElementById('drawModalContent');
  modalContent.innerHTML = '<div class="text-sm text-gray-600">Contacting server…</div>';
  const modal = document.getElementById('drawModal');
  modal.classList.remove('hidden'); modal.style.display='flex';

  try {
    const resp = await fetch('/hiyorimatcha/ajax_draw.php', { method: 'POST' });
    const text = await resp.text();
    let j;
    try { j = JSON.parse(text); } catch(e) { j = { ok:0, msg: text }; }

    if (j.ok && j.won == 1) {
      modalContent.innerHTML = '<div class="text-2xl font-bold text-green-700">Congratulations!</div><div class="mt-3">You won a freebies code:</div><div class="mt-2 font-mono text-lg">' + j.code + '</div>';
      document.getElementById('prizeCode').textContent = j.code;
      document.getElementById('prizeCodeArea').style.display = 'block';
      // optionally reload after short delay to update counts
      setTimeout(()=> location.reload(), 1200);
    } else if (j.ok) {
      modalContent.innerHTML = '<div class="text-xl text-gray-700">' + (j.msg || 'Try again') + '</div>';
      setTimeout(()=> { modal.classList.add('hidden'); modal.style.display=''; }, 2200);
      setTimeout(()=> location.reload(), 1200);
    } else {
      modalContent.innerHTML = '<div class="text-red-600">Error</div><div class="mt-2 text-sm">' + (j.msg || 'Unknown') + '</div>';
    }
  } catch (err) {
    modalContent.innerHTML = '<div class="text-red-600">Network or server error</div><div class="mt-2 text-sm">Please try again later.</div>';
  } finally {
    btn.disabled = false;
  }
});

document.getElementById('drawModalClose').addEventListener('click', ()=>{
  const modal = document.getElementById('drawModal');
  modal.classList.add('hidden'); modal.style.display='';
});
</script>
<script>
  // helper to show simple modal popup
  function showPopup(html) {
    const m = document.getElementById('hiyoModal');
    const content = document.getElementById('modalContent');
    content.innerHTML = html;
    m.classList.remove('hidden');
  }
  document.getElementById('modalClose').addEventListener('click', ()=> {
    document.getElementById('hiyoModal').classList.add('hidden');
  });

  // REDEEM code via redeem.php
  document.getElementById('redeemBtn').addEventListener('click', async () => {
    const btn = document.getElementById('redeemBtn');
    const input = document.getElementById('redeemCodeInput');
    const code = input.value.trim().toUpperCase();

    if (!/^[A-Z0-9]{6}$/.test(code)) {
      showPopup('<h3 class="text-lg font-semibold">Invalid code</h3><p class="mt-2">Please enter a 6 character code like <strong>ABC26N</strong>.</p>');
      return;
    }

    btn.disabled = true;
    btn.classList.add('opacity-70','cursor-wait');

    try {
      const form = new FormData();
      form.append('code', code);

      const res = await fetch('/hiyorimatcha/redeem.php', {
        method: 'POST',
        credentials: 'same-origin',
        body: form
      });
      const j = await res.json();

      if (j.ok) {
        // success
        let msg = '<h3 class="text-xl font-semibold text-[#4A7C59]">Redeemed!</h3>';
        msg += `<p class="mt-3">${j.msg || 'Loyalty stamp added.'}</p>`;
        if (j.special && j.special.voucher_code) {
          msg += `<div class="mt-4 p-3 bg-green-50 rounded"><strong>Voucher:</strong> ${j.special.voucher_code}</div>`;
        }
        showPopup(msg);

        // optional: clear input
        input.value = '';
        // optional: update UI counters - easiest: reload the stamp area or reload the page
        // location.reload(); // uncomment if you prefer automatic refresh
      } else {
        showPopup(`<h3 class="text-lg font-semibold">Not redeemed</h3><p class="mt-2">${j.msg || 'Unknown error'}</p>`);
      }
    } catch (e) {
      showPopup('<h3 class="text-lg font-semibold">Network error</h3><p class="mt-2">Could not reach the server.</p>');
    } finally {
      // small cooldown to avoid accidental double clicks
      setTimeout(()=> {
        btn.disabled = false;
        btn.classList.remove('opacity-70','cursor-wait');
      }, 600);
    }
  });

  // LUCKY DRAW
  const playBtn = document.getElementById('luckyPlayBtn');
  playBtn.addEventListener('click', async () => {
    // client-side quick disable for UX
    playBtn.disabled = true;
    playBtn.classList.add('opacity-70','cursor-wait');

    try {
      const res = await fetch('/hiyorimatcha/play_luckydraw.php', {
        method: 'POST',
        credentials: 'same-origin'
      });
      const j = await res.json();

      if (!j.ok) {
        // server returned an error (not playable)
        showPopup(`<h3 class="text-lg font-semibold">Oops</h3><p class="mt-2">${j.msg || 'Unable to play'}</p>`);
      } else {
        if (j.won == 1) {
          // win
          showPopup(`<h3 class="text-xl font-semibold text-[#4A7C59]">Congratulations!</h3>
                     <p class="mt-2">You won a freebie. Code: <strong>${j.code}</strong></p>
                     <p class="mt-3 text-sm text-gray-600">Please inform staff to redeem this code.</p>`);
          // optionally show it in account page below referal code area, you'll need server refresh or an extra AJAX to fetch codes
        } else {
          // lose
          const left = j.left_to_guarantee ?? '—';
          showPopup(`<h3 class="text-lg font-semibold">Awww, try again</h3>
                     <p class="mt-2">Better luck next time. ${left} more until guaranteed win.</p>`);
        }
      }
    } catch (e) {
      showPopup('<h3 class="text-lg font-semibold">Network error</h3><p class="mt-2">Could not reach server.</p>');
    } finally {
      // server enforces 3s cooldown. client-side re-enable after 3s UX safety.
      setTimeout(()=> {
        playBtn.disabled = false;
        playBtn.classList.remove('opacity-70','cursor-wait');
      }, 3000);
    }
  });
</script>

</body>
</html>
