<?php
// redeem.php (final production version)
// Implements loyalty logic: NEW USER -> TIER -> REFERRAL (priority)
// Generates prize when loyalty_stamps >= 10 and resets to 0
// Returns JSON with updated user counters

declare(strict_types=1);
session_start();
header('Content-Type: application/json; charset=utf-8');

try {
    require_once __DIR__ . '/scripts/db.php'; // must set $pdo (PDO)
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['success'=>false,'message'=>'Server boot error','error'=>$e->getMessage()]);
    exit;
}

if (empty($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success'=>false,'message'=>'Not authenticated']);
    exit;
}
$uid = (int) $_SESSION['user_id'];
$code = isset($_POST['code']) ? strtoupper(trim((string)$_POST['code'])) : null;
$clientIp = $_SERVER['REMOTE_ADDR'] ?? '';

if (!$code || !preg_match('/^[A-Z0-9]{6}$/', $code)) {
    echo json_encode(['success'=>false,'message'=>'Invalid code format']);
    exit;
}

date_default_timezone_set('Asia/Jakarta');
$today = date('Y-m-d');

try {
    $pdo->beginTransaction();

    // lock order_codes
    $stmt = $pdo->prepare("SELECT * FROM order_codes WHERE code = :code FOR UPDATE");
    $stmt->execute([':code' => $code]);
    $codeRow = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$codeRow) {
        $pdo->rollBack();
        echo json_encode(['success'=>false,'message'=>'Code not found']);
        exit;
    }
    if (!array_key_exists('valid_date',$codeRow) || $codeRow['valid_date'] !== $today) {
        $pdo->rollBack();
        echo json_encode(['success'=>false,'message'=>'Code not valid today','valid_date'=>$codeRow['valid_date'] ?? null,'today'=>$today]);
        exit;
    }
    if (array_key_exists('used_by',$codeRow) && !is_null($codeRow['used_by'])) {
        $pdo->rollBack();
        echo json_encode(['success'=>false,'message'=>'Code already used']);
        exit;
    }

    // lock user
    $ust = $pdo->prepare("SELECT * FROM users WHERE id = :uid FOR UPDATE");
    $ust->execute([':uid' => $uid]);
    $user = $ust->fetch(PDO::FETCH_ASSOC);
    if (!$user) {
        $pdo->rollBack();
        echo json_encode(['success'=>false,'message'=>'User not found']);
        exit;
    }

    // read schema columns once
    $colQ = $pdo->prepare("SELECT COLUMN_NAME FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'users'");
    $colQ->execute();
    $userCols = $colQ->fetchAll(PDO::FETCH_COLUMN, 0);

    // required columns validation
    if (!in_array('loyalty_stamps', $userCols, true) || !in_array('xp', $userCols, true)) {
        $pdo->rollBack();
        echo json_encode(['success'=>false,'message'=>'Missing required DB columns: loyalty_stamps and xp']);
        exit;
    }

    // configuration
    $BASE_STAMPS = 3; // testing mode; change to 1 in production
    $TIERS = [
        'first' => ['xp_per'=>10, 'next_xp'=>100, 'chance'=>0, 'max'=>0, 'guarantee'=>0],
        'classic'=> ['xp_per'=>15, 'next_xp'=>500, 'chance'=>10, 'max'=>1, 'guarantee'=>8],
        'ceremonial'=> ['xp_per'=>25, 'next_xp'=>null, 'chance'=>20, 'max'=>2, 'guarantee'=>3],
    ];

    $tier = $user['tier'] ?? 'first';
    if (!isset($TIERS[$tier])) $tier = 'first';
    $tierCfg = $TIERS[$tier];

    $bonus = 0;
    $actions = [];

    // 1) NEW USER BONUS (highest priority)
    $newFlagCol = in_array('new_user_flag',$userCols,true) ? 'new_user_flag' : (in_array('signup_bonus_pending',$userCols,true) ? 'signup_bonus_pending' : null);
    $newFlag = $newFlagCol ? (int)($user[$newFlagCol] ?? 0) : 0;
    if ($newFlag === 1) {
        $bonus += 1;
        $actions[] = 'new_user_bonus';
        // clear flag
        if ($newFlagCol) $pdo->prepare("UPDATE users SET {$newFlagCol} = 0 WHERE id = :uid")->execute([':uid'=>$uid]);

        // credit referrer via signup_referral_code OR referred_by
        if (!empty($user['signup_referral_code'])) {
            $r = $pdo->prepare("SELECT id FROM users WHERE referral_code = :rc LIMIT 1");
            $r->execute([':rc' => $user['signup_referral_code']]);
            $ref = $r->fetch(PDO::FETCH_ASSOC);
            if ($ref) {
                $refid = (int)$ref['id'];
                if (in_array('referral_stamps',$userCols,true)) {
                    $pdo->prepare("UPDATE users SET referral_stamps = referral_stamps + 1 WHERE id = :rid")->execute([':rid'=>$refid]);
                } elseif (in_array('referral_flags',$userCols,true)) {
                    $pdo->prepare("UPDATE users SET referral_flags = referral_flags + 1 WHERE id = :rid")->execute([':rid'=>$refid]);
                }
                $actions[] = "credited_referrer_{$refid}";
            }
        } elseif (!empty($user['referred_by'])) {
            $refid = (int)$user['referred_by'];
            if (in_array('referral_stamps',$userCols,true)) {
                $pdo->prepare("UPDATE users SET referral_stamps = referral_stamps + 1 WHERE id = :rid")->execute([':rid'=>$refid]);
            } elseif (in_array('referral_flags',$userCols,true)) {
                $pdo->prepare("UPDATE users SET referral_flags = referral_flags + 1 WHERE id = :rid")->execute([':rid'=>$refid]);
            }
            $actions[] = "credited_referrer_{$refid}";
        }
    }

    // 2) TIER BONUS
    $cycleUsedCol = in_array('cycle_bonus_used',$userCols,true) ? 'cycle_bonus_used' : null;
    $cycleLossCol = in_array('cycle_loss_counter',$userCols,true) ? 'cycle_loss_counter' : null;
    if ($tier !== 'first' && $cycleUsedCol && $cycleLossCol) {
        $cycleUsed = (int)($user[$cycleUsedCol] ?? 0);
        $cycleLoss = (int)($user[$cycleLossCol] ?? 0);
        if ($cycleUsed < $tierCfg['max']) {
            $rand = random_int(1,100);
            $guarantee = ($tierCfg['guarantee'] > 0 && $cycleLoss >= $tierCfg['guarantee']);
            if ($rand <= $tierCfg['chance'] || $guarantee) {
                $bonus += 1;
                $actions[] = 'tier_bonus';
                $pdo->prepare("UPDATE users SET {$cycleUsedCol} = {$cycleUsedCol} + 1, {$cycleLossCol} = 0 WHERE id = :uid")->execute([':uid'=>$uid]);
            } else {
                $pdo->prepare("UPDATE users SET {$cycleLossCol} = {$cycleLossCol} + 1 WHERE id = :uid")->execute([':uid'=>$uid]);
            }
        } else {
            $pdo->prepare("UPDATE users SET {$cycleLossCol} = {$cycleLossCol} + 1 WHERE id = :uid")->execute([':uid'=>$uid]);
        }
    }

    // 3) REFERRAL BONUS (consumption)
    $referralCol = in_array('referral_stamps',$userCols,true) ? 'referral_stamps' : (in_array('referral_flags',$userCols,true) ? 'referral_flags' : null);
    if ($referralCol) {
        $available = (int)($user[$referralCol] ?? 0);
        if ($available > 0) {
            $bonus += 1;
            $actions[] = 'referral_bonus_used';
            $pdo->prepare("UPDATE users SET {$referralCol} = {$referralCol} - 1 WHERE id = :uid")->execute([':uid'=>$uid]);
        }
    }

    if ($bonus > 2) $bonus = 2;
    $totalAward = $BASE_STAMPS + $bonus;
    $xpGain = $TIERS[$tier]['xp_per'] ?? 10;

    // mark code used
    $pdo->prepare("UPDATE order_codes SET used_by = :uid, used_at = NOW() WHERE id = :id")->execute([':uid'=>$uid, ':id'=>$codeRow['id']]);

    // update user counters
    $pdo->prepare("UPDATE users SET loyalty_stamps = loyalty_stamps + :add, xp = xp + :xp WHERE id = :uid")
        ->execute([':add'=>$totalAward, ':xp'=>$xpGain, ':uid'=>$uid]);

    // add redeem log if exists
    $tables = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN, 0);
    if (in_array('redeem_logs', $tables, true)) {
        try {
            $pdo->prepare("INSERT INTO redeem_logs (user_id, code_id, ip) VALUES (:uid, :codeid, :ip)")
                ->execute([':uid'=>$uid, ':codeid'=>$codeRow['id'], ':ip'=>$clientIp]);
        } catch (Throwable $e) {
            // ignore logging failure
        }
    }

    // -> PRIZE generation: re-check loyalty_stamps FOR UPDATE and generate prize(s) if >=10
    $prizeCreated = null;
    if (in_array('prize_codes', $pdo->query("SELECT TABLE_NAME FROM information_schema.TABLES WHERE TABLE_SCHEMA = DATABASE()")->fetchAll(PDO::FETCH_COLUMN, 0), true)) {
        $stmt2 = $pdo->prepare("SELECT loyalty_stamps FROM users WHERE id = :uid FOR UPDATE");
        $stmt2->execute([':uid'=>$uid]);
        $ls = (int)$stmt2->fetchColumn();
        while ($ls >= 10) {
            // generate unique 6-char code
            $chars = 'ABCDEFGHJKMNPQRSTUVWXYZ23456789';
            $newCode = null;
            for ($i=0;$i<10;$i++){
                $s='';
                for ($k=0;$k<6;$k++) $s .= $chars[random_int(0, strlen($chars)-1)];
                $chk = $pdo->prepare("SELECT id FROM prize_codes WHERE code = :c LIMIT 1");
                $chk->execute([':c'=>$s]);
                if (!$chk->fetch()) {
                    $pdo->prepare("INSERT INTO prize_codes (user_id, code, type, created_at) VALUES (:uid, :c, 'loyalty', NOW())")
                        ->execute([':uid'=>$uid, ':c'=>$s]);
                    $newCode = $s;
                    break;
                }
            }
            if (!$newCode) {
                $newCode = substr(strtoupper(hash('sha256', uniqid((string)random_int(0, PHP_INT_MAX), true))), 0, 6);
                $pdo->prepare("INSERT INTO prize_codes (user_id, code, type, created_at) VALUES (:uid, :c, 'loyalty', NOW())")
                    ->execute([':uid'=>$uid, ':c'=>$newCode]);
            }

            // reset loyalty_stamps to 0 (per spec)
            $pdo->prepare("UPDATE users SET loyalty_stamps = 0 WHERE id = :uid")->execute([':uid'=>$uid]);

            // reset cycle counters for this user
            $resetset = [];
            if (in_array('cycle_bonus_used', $userCols, true)) $resetset[] = "cycle_bonus_used = 0";
            if (in_array('cycle_loss_counter', $userCols, true)) $resetset[] = "cycle_loss_counter = 0";
            if (!empty($resetset)) $pdo->prepare("UPDATE users SET ".implode(', ',$resetset)." WHERE id = :uid")->execute([':uid'=>$uid]);

            $prizeCreated = $newCode;

            // re-read loyalty_stamps for possible multiple cycles
            $stmt2->execute([':uid'=>$uid]);
            $ls = (int)$stmt2->fetchColumn();
        }
    }

    $pdo->commit();

    // fetch fresh user state
    $fresh = $pdo->prepare("SELECT loyalty_stamps, xp, tier, ".(in_array('referral_stamps', $userCols, true) ? 'referral_stamps' : '0 AS referral_stamps')." FROM users WHERE id = :uid LIMIT 1");
    $fresh->execute([':uid'=>$uid]);
    $f = $fresh->fetch(PDO::FETCH_ASSOC);

    // Promotion logic (non-blocking): compute and apply if threshold reached
    $currentTier = $user['tier'] ?? 'first';
    $newTier = $currentTier;
    $currentXp = (int)($f['xp'] ?? 0);
    while (true) {
        if ($newTier === 'first' && $currentXp >= $TIERS['first']['next_xp']) $newTier = 'classic';
        elseif ($newTier === 'classic' && $currentXp >= $TIERS['classic']['next_xp']) $newTier = 'ceremonial';
        else break;
    }
    if ($newTier !== $currentTier) {
        $pdo->prepare("UPDATE users SET tier = :tier WHERE id = :uid")->execute([':tier'=>$newTier, ':uid'=>$uid]);
        $f['tier'] = $newTier;
        $actions[] = 'tier_promoted_to_'.$newTier;
    }

    // final response
    $out = [
        'success' => true,
        'message' => 'Code redeemed',
        'awarded' => ['base'=>$BASE_STAMPS, 'bonus'=>$bonus, 'total'=>$totalAward, 'xp'=>$xpGain],
        'actions' => $actions,
        'user' => [
            'loyalty_stamps' => (int)($f['loyalty_stamps'] ?? 0),
            'xp' => (int)($f['xp'] ?? 0),
            'tier' => $f['tier'] ?? $tier,
            'referral_stamps' => (int)($f['referral_stamps'] ?? 0),
        ],
    ];
    if ($prizeCreated) $out['prize_code'] = $prizeCreated;

    echo json_encode($out);
    exit;

} catch (Throwable $e) {
    if (isset($pdo) && $pdo->inTransaction()) $pdo->rollBack();
    http_response_code(500);
    error_log('[redeem.php] ' . $e->getMessage());
    echo json_encode(['success'=>false,'message'=>'Server error during redeem','error'=>$e->getMessage()]);
    exit;
}
