<?php
// account.php - new UI with Tailwind + server backend (no output before session_start)
declare(strict_types=1);
session_start();
require_once __DIR__ . '/scripts/db.php';

// ensure logged in
if (!isset($_SESSION['user_id']) || empty($_SESSION['user_id'])) {
    header('Location: /auth/login.php');
    exit;
}
$uid = (int) $_SESSION['user_id'];

// Load user (select safe columns)
$colsStmt = $pdo->prepare("SELECT COLUMN_NAME FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'users'");
$colsStmt->execute();
$cols = $colsStmt->fetchAll(PDO::FETCH_COLUMN, 0);
$has_loyalty = in_array('loyalty_stamps', $cols, true);
$has_referral = in_array('referral_stamps', $cols, true);
$has_xp = in_array('xp', $cols, true);
$has_tier = in_array('tier', $cols, true);
$has_new_user = in_array('new_user_flag', $cols, true) || in_array('signup_bonus_pending', $cols, true);

$select = ['id','email','name','phone','referral_code','created_at'];
if (in_array('signup_referral_code',$cols,true)) $select[] = 'signup_referral_code';
if (in_array('referred_by',$cols,true)) $select[] = 'referred_by';
if ($has_new_user) $select[] = in_array('new_user_flag',$cols,true) ? 'new_user_flag' : 'signup_bonus_pending';
if ($has_referral) $select[] = 'referral_stamps';
if ($has_loyalty) $select[] = 'loyalty_stamps';
if ($has_xp) $select[] = 'xp';
if ($has_tier) $select[] = 'tier';
if (in_array('cycle_bonus_used',$cols,true)) $select[] = 'cycle_bonus_used';
if (in_array('cycle_loss_counter',$cols,true)) $select[] = 'cycle_loss_counter';

$selSql = 'SELECT ' . implode(',', $select) . ' FROM users WHERE id = :uid LIMIT 1';
$stmt = $pdo->prepare($selSql);
$stmt->execute([':uid' => $uid]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$user) {
    session_destroy();
    header('Location: /auth/login.php');
    exit;
}

// normalize
$display = [
    'id' => (int)$user['id'],
    'name' => $user['name'] ?? $user['email'],
    'email' => $user['email'] ?? '',
    'phone' => $user['phone'] ?? '',
    'referral_code' => $user['referral_code'] ?? '',
    'signup_referral_code' => $user['signup_referral_code'] ?? null,
    'new_user_flag' => (int)($user['new_user_flag'] ?? $user['signup_bonus_pending'] ?? 0),
    'referral_stamps' => (int)($user['referral_stamps'] ?? 0),
    'loyalty_stamps' => (int)($user['loyalty_stamps'] ?? 0),
    'xp' => (int)($user['xp'] ?? 0),
    'tier' => $user['tier'] ?? 'first',
    'cycle_bonus_used' => (int)($user['cycle_bonus_used'] ?? 0),
    'cycle_loss_counter' => (int)($user['cycle_loss_counter'] ?? 0),
];

// prizes
$prizes = [];
$prizeExists = (bool)$pdo->query("SELECT 1 FROM information_schema.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'prize_codes'")->fetchColumn();
if ($prizeExists) {
    $ps = $pdo->prepare("SELECT id, code, type, claimed, claimed_at, created_at FROM prize_codes WHERE user_id = :uid ORDER BY created_at DESC");
    $ps->execute([':uid' => $uid]);
    $prizes = $ps->fetchAll(PDO::FETCH_ASSOC);
}

// xp thresholds
$xpThreshold = $display['tier'] === 'first' ? 100 : ($display['tier'] === 'classic' ? 500 : 0);
$xpPercent = $xpThreshold>0 ? min(100, (int)round(($display['xp'] / $xpThreshold) * 100)) : 100;
$xpToNext = $xpThreshold>0 ? max(0, $xpThreshold - $display['xp']) : 0;

// produce JSON state for client
$payload = ['user' => $display, 'prizes' => $prizes, 'xp' => ['percent'=>$xpPercent, 'to_next'=>$xpToNext, 'threshold'=>$xpThreshold]];

?><!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8" />
  <title>Account â€¢ Hiyori Artisan Matcha</title>

  <!-- Tailwind CDN (quick) -->
  <script src="https://cdn.tailwindcss.com"></script>
  <meta name="viewport" content="width=device-width,initial-scale=1">

  <!-- Custom theme variables (matcha green #4A7C59) -->
  <style>
    :root{
      --hiyori-green: #4A7C59;
      --accent: #88b27a;
      --muted: #6b7280;
      --card-bg: #ffffff;
      --radius: 12px;
    }
    body { font-family: Inter, ui-sans-serif, system-ui, -apple-system, "Segoe UI", Roboto, "Helvetica Neue", Arial; background: #f8faf6; color: #0f172a; }
    .card { background: var(--card-bg); border-radius: var(--radius); box-shadow: 0 6px 18px rgba(10,10,10,0.06); }
    .btn-primary { background: linear-gradient(180deg,var(--hiyori-green),var(--accent)); color: white; }
    .xp-bar { background:#e6f0e9; height:10px; border-radius:999px; overflow:hidden; }
    .xp-fill { height:100%; background: linear-gradient(90deg,var(--hiyori-green),#7fb77b); width:0%; }
    .small-muted { color:var(--muted); font-size:0.9rem; }
    /* responsive layout */
  </style>
</head>
<body class="min-h-screen flex items-start justify-center p-6">
  <main class="w-full max-w-4xl">
    <header class="mb-6">
      <div class="flex items-center justify-between">
        <div>
          <h1 class="text-2xl font-semibold">Welcome, <?php echo htmlspecialchars($display['name']); ?></h1>
          <div class="small-muted">Member since <?php echo htmlspecialchars(date('F j, Y', strtotime($user['created_at'] ?? 'now'))); ?></div>
        </div>
        <div class="text-right">
          <div class="text-sm small-muted">Referral code</div>
          <div class="mt-1 inline-block px-3 py-2 bg-white card text-sm font-medium"><?php echo htmlspecialchars($display['referral_code']); ?></div>
          <div class="mt-3"><a href="/auth/logout.php" class="text-sm small-muted hover:underline">Logout</a></div>
        </div>
      </div>
    </header>

    <section class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
      <div class="card p-4">
        <div class="text-sm small-muted">Loyalty stamps</div>
        <div class="text-3xl font-bold mt-2"><?php echo (int)$display['loyalty_stamps']; ?></div>
        <div class="small-muted mt-2">Collect 10 stamps to earn a voucher</div>
      </div>
      <div class="card p-4">
        <div class="text-sm small-muted">Referral stamps</div>
        <div class="text-3xl font-bold mt-2"><?php echo (int)$display['referral_stamps']; ?></div>
        <div class="small-muted mt-2">Use when redeeming to get +1 bonus stamp</div>
      </div>
      <div class="card p-4">
        <div class="text-sm small-muted">Tier & XP</div>
        <div class="flex items-center justify-between mt-2">
          <div>
            <div class="text-xl font-semibold"><?php echo htmlspecialchars(ucfirst($display['tier'])); ?></div>
            <div class="small-muted"><?php echo $display['xp']; ?> XP</div>
          </div>
          <div style="width:40%">
            <div class="xp-bar"><div class="xp-fill" id="xp-fill" style="width: <?php echo $xpPercent; ?>%"></div></div>
            <div class="small-muted text-right mt-1"><?php echo $xpPercent; ?>%</div>
          </div>
        </div>
      </div>
    </section>

    <section class="card p-4 mb-6">
      <h2 class="text-lg font-semibold mb-3">Redeem loyalty code</h2>
      <form id="redeem-form" class="flex gap-3 items-center">
        <input id="redeem-code" name="code" maxlength="6" autocomplete="off"
               class="flex-1 px-4 py-3 border rounded-md focus:outline-none" placeholder="Enter 6-char code e.g. ABC27N">
        <button id="redeem-btn" type="button" class="px-5 py-3 rounded-md btn-primary">Redeem</button>
      </form>
      <div id="redeem-message" class="mt-3 text-sm"></div>

      <div class="mt-4 text-xs small-muted">
        Codes valid only on the day printed (WIB). This page posts to <code>/redeem.php</code> and shows the server JSON below.
      </div>

      <div class="mt-4">
        <h3 class="text-sm font-medium mb-2">Raw redeem response (JSON)</h3>
        <pre id="redeem-raw" class="p-3 bg-gray-50 rounded-md text-sm" style="max-height:220px;overflow:auto"></pre>
      </div>
    </section>

    <section class="card p-4 mb-6">
      <h3 class="text-lg font-semibold mb-3">Your prizes</h3>
      <div>
        <?php if (count($prizes) === 0): ?>
          <div class="small-muted">No prizes yet</div>
        <?php else: ?>
          <ul class="space-y-2">
            <?php foreach($prizes as $p): ?>
              <li class="p-3 border rounded-md flex justify-between items-center">
                <div>
                  <div class="font-medium"><?php echo htmlspecialchars($p['code']); ?></div>
                  <div class="small-muted text-xs"><?php echo htmlspecialchars($p['type']); ?> â€¢ <?php echo htmlspecialchars($p['created_at']); ?></div>
                </div>
                <div>
                  <?php if (!$p['claimed']): ?>
                    <span class="px-3 py-1 rounded-full text-sm" style="background:#eef7ee;color:var(--hiyori-green)">Unclaimed</span>
                  <?php else: ?>
                    <span class="small-muted text-sm">Claimed</span>
                  <?php endif; ?>
                </div>
              </li>
            <?php endforeach; ?>
          </ul>
        <?php endif; ?>
      </div>
    </section>

    <footer class="text-center small-muted mb-6">Hiyori Artisan Matcha â€” handcrafted matcha. &nbsp;</footer>
  </main>

<script>
  const state = <?php echo json_encode($payload, JSON_HEX_TAG|JSON_HEX_APOS|JSON_HEX_QUOT); ?>;
  document.getElementById('xp-fill').style.width = state.xp.percent + '%';

async function redeemCode(code) {
  const btn = document.getElementById('redeem-btn');
  const msg = document.getElementById('redeem-message');
  const raw = document.getElementById('redeem-raw');
  btn.disabled = true;
  msg.textContent = 'Processing...';
  raw.textContent = '';
  try {
    const fd = new FormData();
    fd.append('code', code);
    const res = await fetch('redeem.php', { method: 'POST', body: fd, credentials: 'same-origin' });
    const text = await res.text();
    raw.textContent = text;
    let json;
    try { json = JSON.parse(text); } catch (e) {
      msg.textContent = 'Server returned non-JSON response (check logs)';
      btn.disabled = false;
      return;
    }

    if (!json.success) {
      msg.textContent = json.message || 'Redeem failed';
      btn.disabled = false;
      return;
    }

    // Update DOM counters from server
    const u = json.user || {};
    if (u.loyalty_stamps !== undefined) {
      const el = document.getElementById('loyalty-stamps');
      if (el) el.textContent = u.loyalty_stamps;
    }
    if (u.referral_stamps !== undefined) {
      const el = document.getElementById('referral-stamps');
      if (el) el.textContent = u.referral_stamps;
    }
    if (u.xp !== undefined) {
      const xpText = document.getElementById('xp-text');
      if (xpText) xpText.textContent = u.xp + ' / ' + (state.xp.threshold || 'MAX');
      const fill = document.getElementById('xp-fill');
      if (fill) {
        const thr = state.xp.threshold || 100;
        const pct = thr>0 ? Math.min(100, Math.round((u.xp / thr) * 100)) : 100;
        fill.style.width = pct + '%';
      }
    }
    if (u.tier) {
      const t = document.getElementById('tier');
      if (t) t.textContent = u.tier;
    }

    if (json.prize_code) {
      msg.textContent = '🎉 Prize: ' + json.prize_code;
      const list = document.getElementById('prizes-list');
      if (list) {
        const html = '<li class="p-3 border rounded-md flex justify-between items-center"><div><div class="font-medium">' + json.prize_code + '</div><div class="small-muted text-xs">loyalty • now</div></div><div><span class="px-3 py-1 rounded-full text-sm" style="background:#eef7ee;color:var(--hiyori-green)">Unclaimed</span></div></li>';
        list.insertAdjacentHTML('afterbegin', html);
      }
    } else {
      msg.textContent = json.message || 'Redeemed';
    }

  } catch (err) {
    console.error(err);
    if (raw) raw.textContent = String(err);
    if (msg) msg.textContent = 'Network / server error';
  } finally {
    btn.disabled = false;
  }
}

  document.getElementById('redeem-btn').addEventListener('click', () => {
    const code = document.getElementById('redeem-code').value.trim().toUpperCase();
    if (!/^[A-Z0-9]{6}$/.test(code)) {
      document.getElementById('redeem-message').textContent = 'Invalid code format (6 alphanumeric chars)';
      return;
    }
    redeemCode(code);
  });
</script>
</body>
</html>
